#!/usr/bin/perl

# Copyright (C) 2005-2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ShowPairings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn Ordinal);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ShowPairings - implement the C<tsh> ShowPairings command

=head1 SYNOPSIS

  my $command = new TSH::Command::ShowPairings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ShowPairings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub FormatRepeats ($$$$$);
sub initialise ($@);
sub new ($);
sub Run ($$@);
sub ShowAlphaPairings ($$$);
sub ShowBye ($$$);
sub ShowHeader ($$$);
sub ShowRankedPairings ($$$);
sub ShowUnpaired ($$$);

=item $s = FormatRepeats($config, $p, $opp, $r0, $html);

Return a string describing the number of times two players have played each other.

=cut

sub FormatRepeats ($$$$$) {
  my $config = shift;
  my $p = shift;
  my $opp = shift;
  my $r0 = shift;
  my $html = shift;
  my $repeat = $p->CountRoundRepeats($opp, $r0);
  my $s = $repeat <= 1 ? '' 
    : $repeat == 2 ? $config->Terminology('repeat') 
    : $config->Terminology('npeat', $repeat);
  $s = "<span class=repeat>$s</span>" if $html && $s;
  $s = " $s" if $s;
  return $s;
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($@) {
  my $this = shift;

  $this->SUPER::initialise(@_);
  $this->{'help'} = <<'EOF';
Use this command to display the pairings for the specified round
and division.
EOF
  $this->{'names'} = [qw(sp showpairings)];
  $this->{'argtypes'} = [qw(Round Division)];

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($round, $dp) = @_;
  my $config = $tournament->Config();

# my $opt_p = 0;
# if (@$argvp && $argvp->[0] eq '-p') { shift @$argvp; $opt_p = 1; }
  my $round0 = $round-1;

  my $changed = $dp->CheckAutoPair($this->Processor(), $round);
  if ($round0 > $dp->LastPairedRound0()) {
    $tournament->TellUser('enopryet', $dp->Name(), $round);
    return;
    }
# # a kludge formerly used with make-rr.pl
# if ($opt_p) {
#   print '[';
#   print join(',', map { $_->{'pairings'}[$round0]-1 } @$datap[1..$#$datap]);
#   print "]\n";
#   return 0;
#   }
  {
    my $sr0 = $round - 2;
    $sr0 = 0 if $sr0 < 0;
    # shouldn't this be MostScores() - 1?  does it matter? see SMP.pm too.
    $sr0 = $dp->MostScores() if $sr0 > $dp->MostScores()-1;
    $dp->ComputeBoards($sr0, $round0); 
    $this->{'sp_ranked'} = [TSH::Player::SortByStanding $sr0, $dp->Players()];
  }
  $this->{'sp_tables'} = $config->Value('tables')->{$dp->Name()};

  $this->ShowTallySlips($dp, $round);
  $this->ShowRankedPairings($dp, $round);
  $this->ShowAlphaPairings($dp, $round);
  if (my $processor = $this->Processor()) { 
    $processor->Flush(); # in case table numbers were changed
    if ($changed) {
      if (my $cmds = $config->Value('hook_autopair')) {
	$processor->Process($cmds, 'nohistory');
	}
      }
    }
  }

=item ShowAlphaPairings;

Create an HTML file listing pairings in alphabetical order by player name.

=cut

sub ShowAlphaPairings ($$$) {
  my $this = shift;
  my $dp = shift;
  my $round = shift;
  my $round0 = $round - 1;
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $hasphotos = $config->Value('player_photos');
  my $page_break = $config->Value('alpha_pair_page_break');
  my $noboards = $config->Value('no_boards');
  my $single_column = $config->Value('alpha_pair_single_column');
  my $tables = $this->{'sp_tables'};

  my @entries;
  my @headless;
  my @sorted;
  if ($config->Value('sort_by_first_name')) {
    @sorted = sort { $a->PrettyName() cmp $b->PrettyName() } $dp->Players();
    }
  else {
    @sorted = sort { $a->Name() cmp $b->Name() } $dp->Players();
    }
  for my $p (@sorted) {
    next unless $p->Active();
    my $oppid = $p->OpponentID($round0, 'undef for unpaired');
    my $entry = '';
    my $board = $p->Board($round0);
#   die "No board for player $p->{'name'} in round $round0+1\n";
    if (defined $tables) {
      if ($oppid) {
	$entry .= "<td class=table>$tables->[$board-1]</td>";
	}
      else {
	$entry .= "<td class=table>&nbsp;</td>";
	}
      }
    unless ($noboards) {
      if ($oppid) { $entry .= "<td class=board>$board</td>"; }
      else { $entry .= "<td class=board>&nbsp;</td>"; }
      }
    $entry .= "<td>" . $p->TaggedHTMLName('print') . "</td>";
    if (!defined $oppid) {
      $entry .= "<td>".$config->Terminology('Unpaired')."</td>";
      }
    elsif (!$oppid) {
      $entry .= "<td>".$config->Terminology('Bye')."</td>";
      }
    else {
      my $opp = $dp->Player($oppid);
      my ($p12s,$o12s) = $dp->FormatPairing($round0, $p->ID(), 'balanced');
      my $pic = '';
      if ($hasphotos) {
#	warn join("\n", keys %{$tournament->{'bios'}});
#	die $opp->Name();
	my $url = $opp->PhotoURL();
	if ($url) {
          my $aspect = $config->Value('player_photo_aspect_ratio') || 1;
	  my $width = int(0.5+$aspect * 36);
	  $pic = qq(<img align=left height=36 width=$width src="$url">);
	  if ($url =~ /unknown_player/) {
	    push(@headless, $opp);
	    }
	  }
	else {
	  push(@headless, $opp);
	  }
        }
      $entry =~ s/<\/td>$/ $p12s$&/
        if $config->Value('track_firsts');
      $entry .= "<td>$pic" . ($opp->TaggedHTMLName()) .
	($config->Value('track_firsts') ? " $o12s" : '') .
        (FormatRepeats $config, $p, $opp, $round0, 1) . 
        "</td>";
      }
    push(@entries, $entry);
    }
  if ($hasphotos && @headless) {
    my $fn = $config->MakeRootPath($dp->Name() . '-headless.txt');
    if (open my $fh, ">$fn") {
      binmode $fh, ':encoding(isolatin1)';
      if ($noboards) {
	print $fh map { $_->Name() . "\n" } 
	  sort { $a->Name() cmp $b->Name() } @headless;
        }
      else {
	print $fh map { $_->Board($round0) . ' ' . $_->Name() . "\n" } 
	  sort { $a->Board($round0) cmp $b->Board($round0) } @headless;
	}
      close $fh; }
    }
  my $logp = new TSH::Log($tournament, $dp, 'alpha-pairings', $round, {
    'noconsole' => 1,
    'notext' => 1,
    'titlename' => $config->Terminology('Alphabetic_Pairings'),
    },
    );
  my $html = '';
  my $headings = '';
  my $table_term = $config->Value('table_title') || $config->Terminology('Table');
  $headings .= "<th class=table>$table_term</th>" if defined $tables;
  unless ($noboards) {
    $headings .= '<th class=board>'.$config->Terminology('Board').'</th>'
    }
  $headings .= '<th class=name>'.$config->Terminology('Player').'</th>';
# $headings .= "<th class=starts></th>" if $config->Value('track_firsts');
  $headings .= '<th class=name>'.$config->Terminology('Opponent').'</th>';
# $headings .= "<th class=starts></th>" if $config->Value('track_firsts');
# if ($hasphotos < @entries / 5) { # too few photos looks silly
#   for my $entry (@entries) { $entry =~ s/<img.*?>//; }
#   $hasphotos = 0;
#   }
# if ($hasphotos || ($page_break && @entries > $page_break)) {
  if ((!$hasphotos) && ($page_break && @entries > $page_break) && !$single_column) {
#   warn "\@e=".scalar(@entries)." pb=$page_break";
    $html .= "<tr>$headings$headings</tr>";
    push(@entries, '') if @entries % 2;
    my $half = @entries / 2;
    for my $i (0..$half-1) {
      $entries[$i] =~ s/(.*)<td>/$1<td style="border-right:1px solid black">/;
      $html .= qq(<tr class=double style="page-break-inside:avoid">$entries[$i]$entries[$i+$half]</tr>);
      if ($page_break && ($i+1) % $page_break == 0 && $i != $half-1) {
	$html .= qq(</table><table class=alpha_pairings align=center cellspacing=0 style="page-break-before:always"><tr>$headings$headings</tr>);
        }
      }
    }
  else {
    $html .= "<tr>$headings</tr>";
    for my $i (0..$#entries) {
      $html .= qq(<tr class=double style="page-break-inside:avoid">$entries[$i]</tr>);
      if ($page_break && ($i+1) % $page_break == 0 && $i != $#entries) {
	$html .= qq(</table><table class=alpha_pairings align=center cellspacing=0 style="page-break-before:always"><tr>$headings</tr>);
        }
      }
#   $html .= join('', map { qq(<tr style="page-break-inside:avoid">$_</tr>); } @entries);
    }
  $logp->Write('', $html);
  $logp->Close();
  }

=item ShowBye $log, $tables, $p

Used internally to show a bye player

=cut

sub ShowBye ($$$) {
  my $logp = shift;
  my $tables = shift;
  my $p = shift;
  my $config = $p->Division()->Tournament()->Config();
  my $bye_term = $config->Terminology('Bye');

  $logp->Write('', '<tr>');
  $logp->Write(sprintf("  $config'table_format ", ''), 
    '<td class=notable>&nbsp;</td>') if (defined $tables) && !$config->Value('no_boards');
  $logp->Write(
    sprintf("      %s: %s.\n", $p->TaggedName(), $bye_term),
    "<td class=bye>$bye_term</td><td class=name>" . 
    ($p->TaggedName('print')) . '</td>');
  $logp->Write('', '</tr>');
  }

=item ShowHeader $tournament, $log, $tables

Used internally to show the header of the pairings listings

=cut

sub ShowHeader ($$$) {
  my $tournament = shift;
  my $logp = shift;
  my $tables = shift;
  my $config = $tournament->Config();
  my $noboards = $config->Value('no_boards');
  my $board_term = $config->Terminology('Board');
  my $table_term = $config->Value('table_title') || $config->Terminology('Table');
  my $players_term = $config->Terminology('Players');
  my $who_plays_whom_term = $config->Terminology('Who_Plays_Whom');

  $logp->Write('', '<tr class=top1>');
  if (defined $tables) {
    $logp->Write("$table_term ", "<th class=table>$table_term");
    my $shortage = length(sprintf($config::table_format, ''))-3;
    if ($shortage < 0) {
      $tournament->TellUser('wsmtfmt');
      $config::table_format = '%3s';
      $shortage = 0;
      }
    $logp->Write((' ' x $shortage), '</th>');
    }
  unless ($noboards) {
    $logp->Write("$board_term ", "<th class=board>$board_term</th>");
    }
  $logp->Write("$players_term\n", <<"EOF");
<th class=name>$who_plays_whom_term</th>
</tr>
EOF
  }

=item ShowRankedPairings;

Report on pairings in order by current rankings.

=cut

sub ShowRankedPairings ($$$) {
  my $this = shift;
  my $dp = shift;
  my $round = shift;
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $noboards = $config->Value('no_boards');
  my $ranked_pairings_term = $config->Terminology('Ranked_Pairings');

  my $round0 = $round - 1;
  # sample config line: perl $config'tables{'A'} = [1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10]; # (for 20 2-board tables)
  my $tables = $this->{'sp_tables'};
  my %options;
  if ($config->Value('no_ranked_pairings')) {
    $options{'notext'} = 1;
    $options{'nohtml'} = 1;
    }
  if ($this->{'noconsole'}) {
    $options{'noconsole'} = 1;
    }
  $options{'titlename'} = $ranked_pairings_term;
  my $logp = new TSH::Log($tournament, $dp, 'pairings', $round, \%options);
  ShowHeader $tournament, $logp, $tables;

  my @boards;
  my %done;
  my $sortedp = $this->{'sp_ranked'};
  # scan the list of players, display byes right away, queue unpaired
  # players for display after this loop
  my @unpaired;
  for my $p (@$sortedp) {
    next unless $p->Active();
    my $oppid = $p->OpponentID($round0);
    my $pid = $p->ID();
    if (!defined $oppid) {
      push(@unpaired, $p);
      }
    elsif ($oppid == 0) {
      ShowBye $logp, $tables, $p;
      }
    elsif (!$done{$pid}++) {
      my $opp = $dp->Player($oppid);
      if ($pid != $opp->OpponentID($round0)) {
	my $ooid = $opp->OpponentID($round0);
	my $oppoppname = $ooid ? $opp->Opponent($round0)->Name() : 'bye';
	$tournament->TellUser('ebadpair', 
	  $pid, $p->PrettyName(),
	  $oppid, $opp->PrettyName(),
	  $ooid, $oppoppname);
	}
      elsif ($pid == $oppid) {
	$tournament->TellUser('eselpair', $pid, $p->PrettyName());
	}
      else {
	$done{$oppid}++;
	my $ob = $opp->Board($round0) || 0;
	my $pb = $p->Board($round0) || 0;
	if ($ob != $pb) {
	  $tournament->TellUser('eboarddiff', $p->PrettyName(), $opp->PrettyName());
	  }
	$boards[$pb] = [$p, $opp];
	}
      }
    } # for $p
  # display the queued up unpaired players
  for my $p (@unpaired) {
    ShowUnpaired $logp, $tables, $p;
    }
  # display who's at each board
  # $tables is zero-indexed, but players prefer one-indexed tables
  for my $board1 (1..$#boards) {
    my $board = $board1 - 1;
    my $boardp = $boards[$board1];
    # a board might be empty if a pairing was deactivated after
    # boards were assigned, or if all seating is reserved
    next unless $boardp; 
    $logp->Write('', '<tr>');
    $logp->Write(sprintf(" $config'table_format  ", $tables->[$board]),
      "<td class=table>$tables->[$board]</td>")
      if defined $tables;
    my ($p1, $p2) = @$boardp;
    if ($p1->Board($round0)) {
      $p2->Board($round0, $p1->Board($round0)); # just in case
      }
    else {
      $p1->Board($round0, $board1);
      $p2->Board($round0, $board1);
      }
    my $vs = $dp->FormatPairing($round0, $p1->ID());
    unless ($vs) {
      my $pname = $p1->Name();
      $logp->Write("Lost track of $pname", "Lost track of $pname");
      next;
      }
    my $vshtml = $vs;
    if ($config->Value('track_firsts')) {
      $vshtml =~ s/\*(?:starts|draws)\*/<span class=starts>$&<\/span>/;
      }
    else {
      $vshtml =~ s/\s*\*(?:starts|draws)\*\s*/ /;
      $vs =~ s/\s*\*(?:starts|draws)\*\s*/ /;
      }
    $logp->Write(sprintf(" %3d  ", $board1),
      '<td class=board>' . ($board1) . "</td>"
      ) unless $noboards;
    $logp->Write(sprintf("%s.\n", $vs . (FormatRepeats $config, $p1, $p2, $round0, 0)), "<td class=name>$vshtml"
      . (FormatRepeats $config, $p1, $p2, $round0, 1) . "</td></tr>\n");
    }
  $logp->Close();
  }

=item ShowTallySlips

Generate prefilled tally slips for this round.

=cut

sub ShowTallySlips ($$$) {
  my $this = shift;
  my $dp = shift;
  my $round = shift;
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $noboards = $config->Value('no_boards');
  my $track_firsts = $config->Value('track_firsts');
  my $page_break = $config->Value('tally_slips_page_break') || 9;
  my $blanks = $config->Value('tally_slips_blanks');
  my $challenges = $config->Value('tally_slips_challenges');

  my $round0 = $round - 1;
  my $has_tables = $dp->HasTables();
  my %options;
  $options{'noconsole'} = 1;
  $options{'notitle'} = 1;
  $options{'notop'} = 1;
  my $logp = new TSH::Log($tournament, $dp, 'tally-slips', $round, \%options);
  $logp->Write('', '<tr><td>');

  my %done;
  my $sortedp = [sort { ($a->Board($round0)||0) <=> ($b->Board($round0)||0) } @{$this->{'sp_ranked'}}];
  my @head;
  push(@head, '<span class=division>' . $config->Terminology('Division') . ' ' . $dp->Name() . '</span>')
    if $tournament->CountDivisions() > 1;
  push(@head, '<span class=round>' . $config->Terminology('Round') . ' ' . $round . '</span>');
  my $i = 0;
  for my $p (@$sortedp) {
    next unless $p->Active();
    my $oppid = $p->OpponentID($round0);
    my $pid = $p->ID();
    my $opp = $p->Opponent($round0);
    next unless defined $oppid;
    next if $done{$pid}++;
    $done{$oppid}++;
    if ($track_firsts && $opp && $p->First($round0) == 2) {
      ($pid, $oppid) = ($oppid, $pid);
      ($p, $opp) = ($opp, $p);
      }
    my $board = $p->Board($round0);
    my $table = $has_tables ? '<span class=table>' . $config->Terminology('Table') . ' ' . $dp->BoardTable($board) . '</span>' : '';
    $board = $noboards ? '' : '<span class=board>' . $config->Terminology('Board') . ' ' . $board . '</span>';
    if ($i) {
      if ($page_break && $i % $page_break == 0) { 
	$logp->PageBreak(); 
	}
      else 
        { $logp->Write('', '</table><hr>' . $logp->RenderOpenTable('')); }
      }
    $logp->Write('', <<EOF
<tr class=row1><td class=ident colspan=2>@head $table $board</td>
<th class=score>Score</th>
<th class=initials>Initials</th>
<th class=skip colspan=3>&nbsp;</th>
<th class=score>Score</th>
<th class=initials>Initials</th>
<th class=skip>&nbsp;</th>
<th class=spread>Spread</th>
</tr>
EOF
    );
    $logp->Write('', '<tr class=row2>');
    $logp->Write('', '<td class=p12>' . ($p->First($round0) == 1 ? '1st' : '&nbsp;') . '</td>');
    $logp->Write('', '<td class=player>' . $p->TaggedName() . '</td>');
    $logp->Write('', '<td class=score>&nbsp;</td><td class=initials>&nbsp;</td><td class=skip>&nbsp;</td>');
    if ($opp) {
      $logp->Write('', '<td class=p12>' . ($opp->First($round0) == 2 ? '2nd' : '&nbsp;') . '</td>');
      $logp->Write('', '<td class=player>' . $opp->TaggedName() . '</td>');
      $logp->Write('', '<td class=score>&nbsp;</td><td class=initials>&nbsp;</td><td class=skip>&nbsp;</td><td class=spread>&nbsp;</td>');
      }
    $logp->Write('', '</tr>');
    if ($blanks) {
      my $alphabet = join(' ', 'A'..'Z');
      $logp->Write('', '<tr><td colspan=9><div class=blanks><span class=label>Blanks: </span>'.join('<span class=spacer>&nbsp;&nbsp;</span>',map { "<span class=alphabet$_>$alphabet</span>" } (1..2)).'</div></td></tr>');
      }
    if ($challenges) {
      $logp->Write('', '<tr class=challenges><td colspan=9>Word(s) challenged:</td></tr>');
      }
    $i++;
    }
  $logp->Close();
  }

=item ShowUnpaired $log, $tables, $p

Used internally to show an unpaired player

=cut

sub ShowUnpaired ($$$) {
  my $logp = shift;
  my $tables = shift;
  my $p = shift;
  my $config = $p->Division()->Tournament()->Config();
  my $unpaired_term = $config->Terminology('Unpaired');

  $logp->Write('', '<tr>');
  $logp->Write(sprintf("  $config'table_format ", ''), 
    '<td class=notable>&nbsp;</td>') if defined $tables;
  $logp->Write(sprintf("      %s: $unpaired_term.\n", ($p->TaggedName())),
    "<td class=unpaired>$unpaired_term</td><td class=name>" . 
    ($p->TaggedName('print')) . '</td>');
  $logp->Write('', '</tr>');
  }

=back

=cut

=head1 BUGS

A reasonable guess should be made when there are not enough tables
configured to accommodate the boards needed.

enopryet should be reported even when only inactive players are paired

Should use new TSH::Log table code.

Should use TSH::Division::BoardTable() and TSH::Division::HasTables().

=cut

1;
