#!/usr/bin/perl

# Copyright (C) 2005-2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::UPSETs;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::UPSETs - implement the C<tsh> UPSETs command

=head1 SYNOPSIS

  my $command = new TSH::Command::UPSETs;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::UPSETs is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the UPSETs command to list biggest ratings upsets in a division.
If you specify an integer, it indicates the number of scores you
would like to see.  If not, 20 scores are shown.
EOF
  $this->{'names'} = [qw(upset upsets)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $nscores = @_ ? shift @_ : 20;

  my $logp = new TSH::Log($tournament, $dp, 'upsets');
# warn "1 ".($logp->{'options'}{'noconsole'}||0).' '.($tournament->Config->Value('no_console_log')||0)."\n";
  $logp->ColumnClasses([qw(rating rating rating round score score name name)]);
  $logp->ColumnTitles({
    'text' => [qw(Diff Rtg1 Rtg2 Rd Sc1 Sc2 Player1 Player2)],
    'html' => [
      'Rating<br>Difference',
      'Rating<br>#1','Rating<br>#2',
      'Round',
      'Score<br>#1','Score<br>#2',
      'Player<br>#1','Player<br>#2',
      ],
    });

  # rated upsets
  my $entriesp = TSH::Division::FindExtremeGames::Search($dp, $nscores, 
    # Games are represented as lists: [$score1, $score2, $p1, $p2, $r0, $rat1, $rat2].
    sub ($) { 
      (defined $_[0][0]) # player one has a score
      && (defined $_[0][1]) # player two has a score
      && $_[0][0] > $_[0][1] # player one won
      && $_[0][5] # player one is rated
      && $_[0][5] < $_[0][6] # player one is rated lower than player two
      },
    sub ($$) { $_[1][6]-$_[1][5] <=> $_[0][6]-$_[0][5] }, # big ratings differences first
    );

  for my $entry (@$entriesp) {
    $logp->WriteRow([
      sprintf("%g", $entry->[6]-$entry->[5]),
      $entry->[6],
      $entry->[5],
      $entry->[4]+1,
      $entry->[1],
      $entry->[0],
      $entry->[3]->TaggedName(),
      $entry->[2]->TaggedName(),
      ]);
    }

  # unrated upsets
  $entriesp = TSH::Division::FindExtremeGames::Search($dp, $nscores, 
    # Games are represented as lists: [$score1, $score2, $p1, $p2, $r0, $rat1, $rat2].
    sub ($) { 
      (defined $_[0][0]) # player one has a score
      && (defined $_[0][1]) # player two has a score
      && $_[0][0] > $_[0][1] # player one won
      && (!$_[0][5]) # player one is unrated
      && $_[0][5] < $_[0][6] # player one is rated lower than player two
      },
    sub ($$) { $_[1][6]-$_[1][5] <=> $_[0][6]-$_[0][5] }, # big ratings differences first
    );

  for my $entry (@$entriesp) {
    $logp->WriteRow([
      $entry->[6]-$entry->[5],
      $entry->[6],
      $entry->[5],
      $entry->[4]+1,
      $entry->[1],
      $entry->[0],
      $entry->[3]->TaggedName(),
      $entry->[2]->TaggedName(),
      ]);
    }

  $logp->Close();
  }

=back

=cut

1;
