#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Log;

use strict;
use warnings;

use threads::shared;
use TSH::Utility qw(Error OpenFile PrintConsoleCrossPlatform);
use File::Copy; # copy
use File::Path; # mkpath
use File::Spec; # catfile

=pod

=head1 NAME

TSH::Log - Record displayed information in text and HTML format

=head1 SYNOPSIS

  # original interface
  my $log = new TSH::Log($tournament, $division, $type, $round, \%options);
  $log->Write($text, $html);
  $log->Close();

  # newer interface, dynamically measured columns
  my $log = new TSH::Log($tournament, $division, $type, $round, \%options);
  $log->ColumnClasses([qw(name score)]);
  $log->ColumnTitles({ 'text' => [qw(P S)], 'html' => [qw(Player Score)] });
  $log->ColumnAttributes(['id=1','id=s1']);
  $log->WriteRow(['jjc', 123], ['John Chew (#1)', 123]);
  $log->Close();

  # even newer interface, multiline titles and paging
  my $log = new TSH::Log($tournament, $division, $type, $round, \%options);
  $log->ColumnClasses([qw(name score)],'rowclass');
  $log->ColumnAttributes(['rowspan=2',]);
  $log->WriteTitle([qw('P S')], [qw('Player Score')]);
  $log->ColumnAttributes(['id=1','id=s1']);
  $log->WriteRow(['jjc', 123], ['John Chew (#1)', 123]);
  my ($text, $html) = $log->Close();

  # or instead of Close(), we can call RenderHTML() or RenderText(),
  # see ShowDivisionScoreCards

=head1 ABSTRACT

This class manages the copying of information displayed on the
console to text and HTML files.

=cut

=head1 DESCRIPTION

=head2 FORMAT

A log, in the current version of an increasingly ornate structure, consists
of some or all of the following sequence of sections, which is typically
rendered as text (with automatically sized columns) or HTML.

=over 4

=item header

(HTML only) 
everything beginning with the DOCTYPE and/or HTML open tag
and including the BODY open tag.  A TITLE tag will mark the log's title.

=item top

(HTML only) the optional contents of C<config html_top>.

=item title

The title is based by default on the log type (ratings, scorecard, etc.),
division name (if any) and round number (if any), and can be overridden
by an option.

=item table heading rows

Headings for each of the columns in the table.

=item table data rows

The main content of the log, in table form.

=item footer

(HTML only) the rest of the file, from the tsh blurb down to the HTML close tag.

=back

=head2 METHODS

=over 4

=cut

sub initialise ($$$$;$$);
sub Close($);
sub ColumnAttributes($;$);
sub ColumnClasses($;$$);
sub ColumnTitles($$);
sub EndPage ($$);
sub ExpandEntities ($);
sub Flush ($);
sub GetBlurb ($);
sub new ($$$$;$$);
sub MakeHTMLIndex ($);
sub PageBreak ($);
sub RenderFooterHTML ($$);
sub RenderFooterText ($);
sub RenderHeaderHTML ($$$$);
sub RenderHeaderText ($$$$);
sub RenderHTML ($);
sub RenderOpenTable ($$;$);
sub RenderRowsHTML ($);
sub RenderRowsText ($);
sub RenderText ($);
sub Write ($$$);
sub WriteRow ($$;$);

=item $log->Close()

Close a log and all files associated with it.

=cut

sub Close ($) {
  my $this = shift;
  my $config = $this->{'tournament'}->Config();
  $this->EndPage('last');

  for my $fh (qw(text_fh subhtml_fh)) {
    if ($this->{$fh}) {
      close($this->{$fh});
      $this->{$fh} = undef;
      }
    }
  unless ($this->{'options'}{'nohtml'}) {
    my $target_fn = $config->MakeHTMLPath($this->{'filename'});
    rename "$target_fn.new", $target_fn
      or warn "Can't update $target_fn: $!\n";
    $this->MakeHTMLIndex();
    $this->{'tournament'}->LastReport($this->{'filename'})
      unless $config->Value('nohistory');
    for my $mirror_fn (@{$this->{'mirror_filenames'}}) {
      copy($target_fn, $mirror_fn)
        or warn "Can't update mirror $mirror_fn: $!\n";
      }
    }
  }

=item $p->ColumnAttributes(\@attributes);

Set or get the list of column attributes.

=cut

sub ColumnAttributes ($;$) { TSH::Utility::GetOrSet('column_attributes', @_); }

=item $p->ColumnClasses(\@classes[, row_class]);

Set or get the list of column classes.

=cut

sub ColumnClasses ($;$$) { 
  my $this = shift;
  my $new = shift;
  if (defined $new) {
    $this->{'column_classes'} = $new;
    $new = shift;
    if (defined $new) {
      $this->{'row_class'} = $new;
      }
    else {
      $this->{'row_class'} = undef;
      }
    }
  else {
    return $this->{'column_classes'};
    }
  }

=item $p->ColumnTitles({'text'=>\@text,'html'=>\@html});

Set or get the list of column titles.

=cut

sub ColumnTitles ($$) { 
  my $this = shift;
  my $hashp = shift;
  $this->WriteTitle($hashp->{'text'}, $hashp->{'html'});
  }

=item $log->EndPage($islast);

Emit a footer and end the current page.

=cut

sub EndPage ($$) {
  my $this = shift;
  my $islast = shift;
  $this->Flush();
  $this->Write('', "</table>\n");
  my $text = $this->RenderFooterText();
  my $html = $this->RenderFooterHTML({'nowrapper'=>!$islast});
  $this->Write($text, $html);
  }

sub ExpandEntities ($) {
  $_[0] = '' unless defined $_[0];
  $_[0] =~ s/&([aAoOuU])uml;/$1e/;
  }

=item $action_taken = $p->Flush();

If WriteRow() has been called since the last call to Flush(),
then emit all text and HTML data that has been saved up using
WriteTitle() (or ColumnTitles()) and WriteRow().
Returns boolean indicating whether flushing took place.

=cut

sub Flush ($) {
  my $this = shift;
  return 0 unless @{$this->{'data_text'}} || @{$this->{'data_html'}};
  my $fh;
  my $s = $this->RenderRowsText();
  if ($this->{'options'}{'noconsole'}) {
#   warn "no console for $this->{'base_filename'}";
    }
  else {
    PrintConsoleCrossPlatform $s;
    }
  if ($fh = $this->{'text_fh'}) {
    print $fh $s;
    }
  $s = $this->RenderRowsHTML();
  if ($fh = $this->{'subhtml_fh'}) {
    print $fh $s;
    }
  $this->{'data_text'} = [];
  $this->{'data_html'} = [];
  return 1;
  }

=item $log->initialise($tournament, $division, $filename, $round, \%options)

Initialise the log object, create files.
Options (and defaults): 

htmltitle (title): version of title to use only in HTML format

noconsole (config no_console_log): do not write to console

nohtml (0): do not generate HTML files

notext (config no_text_files): do not generate text file 

notitle (0): do not display title except in HTML <title>

notop (0): do not emit config html_top

nowrapper (0): do not emit (HTML) header or footer

title (Division ? Round ? titlename): human-readable title

titlename (ucfirst $filename): base word to use in human-readable title

texttitle (title): version of title to use only in text format

=cut

sub initialise ($$$$;$$) {
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;
  my $filename = lc shift;
  my $round = lc (shift||'');
  my $optionsp = $this->{'options'} = (shift||&share({}));

  my $config = $tournament->Config();
  my $round_term = $config->Terminology('Round');

  $optionsp->{'noconsole'} ||= $config->Value('no_console_log');

  $this->{'base_filename'} = $filename;
  $this->{'column_attributes'} = [];
  $this->{'column_classes'} = [];
  $this->{'column_titles'} = [];
  $this->{'data_html'} = [];
  $this->{'data_text'} = [];
  $this->{'filename'} = undef;
  $this->{'mirror_filenames'} = [];
  $this->{'row_class'} = undef;
  $this->{'title_html'} = [];
  $this->{'title_text'} = [];
  $this->{'tournament'} = $tournament;

  my ($fh, $fn, %fhs);
  my $round_000 = 
    $round =~ /^\d+$/ ? sprintf("-%03d", $round) 
    : $round =~ /^(\d+)-(\d+)$/ ? sprintf("-%03d-%03d", $1, $2) 
    : '';

  my $dname = '';
  my $dname_hyphen = '';
  if ($dp) {
    $dname = $dp->Name();
    $dname_hyphen = "$dname-";
    }
  unless ($optionsp->{'notext'} || $config->Value('no_text_files')) {
    $fn = $config->MakeRootPath("$dname_hyphen$filename.doc");
    $fh = OpenFile('>:encoding(utf8)', $fn) or Error "Can't create $fn: $!\n"; 
    $this->{'text_fh'} = $fh;
    }

  unless ($optionsp->{'nohtml'}) {
    $this->{'filename'} = "$dname_hyphen$filename$round_000.html";
    if ($config->Value('html_in_event_directory')) {
      push(@{$this->{'mirror_filenames'}}, 
        $config->MakeRootPath("$dname_hyphen$filename.html"));
      }
    for my $mirrordir (@{$config->Value('mirror_directories')}) {
      push(@{$this->{'mirror_filenames'}}, 
	File::Spec->catfile($mirrordir, 'html', $this->{'filename'}));
      mkpath(File::Spec->catfile($mirrordir, 'html'));
      }
    $fn = $config->MakeHTMLPath($this->{'filename'}.'.new');
    $fh = OpenFile('>:encoding(utf8)', $fn) or Error "Can't create $fn: $!\n";
    $this->{'subhtml_fh'} = $fh;
    }

  my $title = $optionsp->{'title'};
  unless (defined $title) {
    $title = ucfirst($optionsp->{'titlename'} || $filename);
    my $i18n;
    eval { $i18n = $config->Terminology($title); };
    $title = $i18n || $title;
    $title = $config->Terminology('Round_Title', $round, $title) if $round =~ /^\d+$/;
    if ($tournament->CountDivisions() > 1 || $config->Value('show_divname')) { 
      $title = $dp ? $dp->Label() . " $title": $title;
      };
    }
  $this->{'title'} = $title;
  my $html = $this->RenderHeaderHTML($title, $filename, $optionsp);
  my $text = $this->RenderHeaderText($title, $filename, $optionsp);
  $this->Write($text, $html);

  my $filename_ = $filename; $filename_ =~ s/-/_/g;
  $this->Write('', $this->RenderOpenTable(''));

  return $this;
  }

=item $fmtsp = $log->MeasureColumns();

Used internally to measure text column widths.

=cut

sub MeasureColumns ($) {
  my $this = shift;
  my @lengths;
  for my $section_key (qw(title_text data_text)) {
    my $rows = $this->{$section_key};
    next unless @$rows;
    for (my $rowi = 1; $rowi <= $#$rows; $rowi += 2) { # no metadata this pass
      my $row = $rows->[$rowi];
      for my $i (0..$#$row) {
	my $cell = $row->[$i];
	next unless defined $cell;
	my $width = length($cell);
	if (defined $lengths[$i]) {
	  $lengths[$i] = $width if $lengths[$i] < $width;
	  }
	else {
	  $lengths[$i] = $width;
	  }
	}
      }
    }

  return \@lengths;
  }

=item $fmt = $log->MakeFormat($length, $class);

Return a sprintf format suitable for data of maximum length $length
and class $class.

=cut

my (%gClassAlignment) = (
  'board' => 'r',
  'onum' => 'r',
  'orat' => 'r',
  'player' => 'l',
  'rank' => 'r',
  'rating' => 'r',
  'rating difr' => 'r',
  'rating newr' => 'r',
  'rating oldr' => 'r',
  'round' => 'r',
  'score' => 'r',
  'sopew' => 'r',
  'spread' => 'r',
  'stat' => 'r',
  'table' => 'r',
  'thai' => 'r',
  'wins' => 'r',
  );

sub MakeFormat ($$) {
  my $length = shift;
  my $class = shift;

  (($gClassAlignment{$class||''}||'') eq 'r') 
    ? "%${length}s"
    : "%-${length}s";
  }

=item $log->MakeHTMLIndex();

Used internally to update the HTML index of HTML files.

=cut

sub MakeHTMLIndex ($) {
  my $this = shift;
  my $dir;
  my $tournament = $this->{'tournament'};
  my $config = $tournament->Config();
  my $round_term = $config->Terminology('Round');
  my $dn = $config->MakeHTMLPath('');
  unless (opendir($dir, $dn)) {
    TSH::Utility::Error "Can't open $dn: $!\n";
    return;
    }
  my @rounds;
  my $max_rounds = $config->Value('max_rounds');
  $#rounds = $max_rounds if $max_rounds;
  my %divisions;
  my $found = 0;
  my %top;
  my $no_index_tally_slips = $config->Value('no_index_tally_slips');
  for my $file (readdir $dir) {
    if ($file =~ /^(rosters|prizes|stats|(?:roto|total_teams)(?:-\d+)?)\.html$/i) {
      $top{$1} = $file;
      $found = 1;
      }
    next if $file =~ /(?:grid|handicaps|ratings|scorecard|standings)\.html/;
    next if $file =~ /tally-slips/ && $no_index_tally_slips;
    next unless $file =~ /^(\w+)-([-a-z]+(?:[^-\d]\d+)?)(?:-(\d+))?(?:-(\d+))?\.html$/i;
    my $div = lc $1;
    my $type = $2;
    my $round = ($3 || 0);
    my $lastround = $4;
    next unless $tournament->GetDivisionByName($div);
    $divisions{$div}++;
    $type = "$type $round-$lastround" if $lastround;
    $round = 0 if $file =~ /\bscoreboard\b/;
    $rounds[$round]{$div}{$type} = $file;
    $found = 1;
    }
  closedir $dir;
  my (@divisions) = sort keys %divisions;
  
  my $fh;
  my $fn = $config->MakeHTMLPath('index.html');
  unless (open $fh, ">$fn") {
    Error "Can't create HTML index file: $!\n";
    return;
    }
  binmode $fh, ':encoding(utf8)';
  print $fh $this->RenderHeaderHTML($config->Terminology('event_coverage_index'), 'index', {'isindex' => 1});
  if ($found) {
    print $fh "<table class=index align=center>\n";
    if (%top) {
      printf $fh "<tr><td class=links colspan=%d align=center>", scalar(@divisions)+1;
      for my $type (sort keys %top) {
        my $i18n = $type;
        eval { 
	  my ($base, $round) = split(/-/, $type, 2);
	  $i18n = $config->Terminology($base); 
	  if ($round) {
	    $i18n = $config->Terminology('Round_Title', $round, $i18n);
	    }
  	  };
	print $fh qq(<div class=link><a href="$top{$type}">$i18n</a></div>\n);
        }
      print $fh "</td></tr>";
      }
    if (@divisions > 1) {
      print $fh "<tr><th class=empty>&nbsp;</th>";
      for my $div (@divisions) {
	print $fh "<th class=division>Div. \U$div\E</th>\n";
	}
      print $fh "</tr>\n";
      }
    for my $round (0..$#rounds) {
      my $rp = $rounds[$round];
      next unless $rp || $max_rounds;
      if ($round) {
	print $fh "<tr><td class=round>$round_term $round</td>\n";
	}
      else {
	print $fh "<tr><td class=round>&nbsp;</td>\n";
	}
      for my $div (@divisions) {
	print $fh "<td class=links>";
	my $rdp = $rp->{$div};
	if ($rdp) {
	  for my $type (sort keys %$rdp) {
	    my $i18n = $type;
 	    eval { $i18n = $config->Terminology($type); };
#	    $i18n = $config->Terminology($type); 
#	    print STDERR "TYPE: $type I18N: $i18n\n";
	    print $fh qq(<div class=link><a href="$rdp->{$type}">$i18n</a></div>\n);
	    }
	  }
	print $fh "</td>";
	}
      print $fh "</tr>\n";
      }
    print $fh "</table>\n";
    }
  else {
    print $fh "<p class=notice>No reports have been generated yet.</p><p>&nbsp;</p>\n";
    }
  print $fh "<p class=notice>\n" . $config->Terminology('links_above') . "</p>\n";
  print $fh $this->RenderFooterHTML({'isindex' => 1});
  close $fh;
  for my $mirrordir (@{$config->Value('mirror_directories')}) {
    copy($fn, File::Spec->catfile($mirrordir, 'html', 'index.html'))
      or warn "Can't mirror index file to $mirrordir: $!";
    }
  }

=item $d = new Log;

Create a new Log object.  

=cut

sub new ($$$$;$$) { return TSH::Utility::new(@_); }

=item $log->PageBreak();

Emit a page footer and break in the HTML version of the report.

=cut

sub PageBreak ($) {
  my $this = shift;
  $this->EndPage(0);
  $this->Write('', $this->RenderOpenTable('style=page-break-before:always',
    'noscreenhead'));
  }

=item $html= $log->RenderFooterHTML(\%options);

Render a standard report trailer in HTML.

=cut

sub RenderFooterHTML ($$) {
  my $this = shift;
  my $optionsp = shift;
  return $optionsp->{'nowrapper'} ? '' 
    : ($this->GetBlurb($optionsp) . "</body></html>");
  }

=item $html = $log->GetBlurb([\%options]);

Return the tsh blurb that appears in the footer.

=cut

sub GetBlurb ($) {
  my $this = shift;
  my $optionsp = shift || {};

  my $config = $this->{'tournament'}->Config();
  my $bottom;
  if ($optionsp->{'isindex'}) {
    $bottom = $config->Value('html_index_bottom');
    $bottom = $config->Value('html_bottom') unless defined $bottom;
    }
  else {
    $bottom = $config->Value('html_bottom');
    }
  $bottom = '' unless defined $bottom;
  return "$bottom<p class=notice>" . $config->Terminology('blurb', $main::gkVersion) . "</p>";
  }

=item $html= $log->RenderFooterText();

Render a standard report trailer in text.

=cut

sub RenderFooterText ($) {
  my $this = shift;
  return '';
  }

=item $html = $log->RenderHTML();

Render entire log (header, table and footer) as HTML and reset table data.

=cut

sub RenderHTML ($) {
  my $this = shift;
  my $html = '';
  $html .= $this->RenderHeaderHTML($this->{'title'},
    $this->{'base_filename'}, $this->{'options'});
  $html .= $this->RenderRowsHTML();
  $html .= $this->RenderFooterHTML($this->{'options'});
  return $html;
  }

=item $html = $log->RenderHeaderHTML($title, $type, \%options);

C<$options{'isindex'}> uses C<$config::html_index_top> 
in preference to C<$config::html_top>.

See initialise() for other options.

=cut

sub RenderHeaderHTML ($$$$) {
  my $this = shift;
  my $title = shift;
  my $type = shift;
  my $optionsp = shift;
# use Carp qw(cluck); cluck "*** $title;$type;$optionsp;$optionsp->{'nowrapper'}";
  my $config = $this->{'tournament'}->Config();
  my $realm = $config->Value('realm');
  $realm = $realm ? " $realm" : '';
  my $html = '';
  my $type_; 
  if ($optionsp->{'body_class'}) { $type_ = $optionsp->{'body_class'} }
  else {
    $type_ = $type;
    $type_ =~ s/-\d+//g;
    $type_ =~ s/-/_/;
   }
  my $custom_stylesheet = $config->Value('custom_stylesheet');
  $custom_stylesheet = defined $custom_stylesheet
    ? qq(<link rel="stylesheet" href="$custom_stylesheet" type="text/css">\n)
    : '';
  my $backref = $optionsp->{'isindex'} ? '' 
    : "<p class=backref>" . $config->Terminology('Back_to_round_index') 
      . ".</p>\n";
   
# use Carp; confess unless defined $title;
  my $script = '';
  if (my $js = $optionsp->{'javascript'}) {
    $js = [$js] unless ref($js) eq 'ARRAY';
    $script = join('', map { qq(<script type="text/javascript" src="$_"></script>) } @$js);
    }
  my $refresh = $optionsp->{'refresh'} ?
    qq(<meta http-equiv=refresh content="$optionsp->{'refresh'}; url=$this->{'filename'}">\n) : '';
  $html .= $optionsp->{'nowrapper'} ? '' : <<"EOF";
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN"
	"http://www.w3.org/TR/1998/REC-html40-19980424/loose.dtd">
<html><head><title>$title</title>
<meta http-equiv="Content-Type" content="text/html; charset=utf-8">
$script$refresh<link rel="stylesheet" href="tsh.css" type="text/css">
$custom_stylesheet</head>
<body class="${type_}$realm">
$backref
EOF
  my $extra_top = $optionsp->{'notop'} ? '' : 
    $optionsp->{'isindex'} 
      ? ($config->Value('html_index_top')||$config->Value('html_top'))
      : $config->Value('html_top');
  $html .= $extra_top if $extra_top;
  my $htmltitle = $optionsp->{'htmltitle'};
  if (!defined $htmltitle) {
    $htmltitle = $title; $htmltitle =~ s/(\d)-(\d)/$1&ndash;$2/g;
    }
  if (length($htmltitle)) { $htmltitle = "<h1>$htmltitle</h1>\n"; }
  return $optionsp->{'notitle'} ? $html : "$html$htmltitle";
  }

=item $html = $log->RenderHeaderText($title, $type, \%options);

See initialise() for options.

=cut

sub RenderHeaderText ($$$$) {
  my $this = shift;
  my $title = shift;
  my $type = shift;
  my $optionsp = shift;
  my $config = $this->{'tournament'}->Config();
  my $type_ = $type; $type =~ s/-/_/;
  return $optionsp->{'notitle'} ? '' :
    ((defined $optionsp->{'texttitle'}) 
      ? $optionsp->{'texttitle'} : "$title\n\n");
  }

=item $html = $log->RenderOpenTable($attrs);

Render an open-table tag for our log.

=cut

sub RenderOpenTable ($$;$) {
  my $this = shift;
  my $attrs = shift;
  my $extra_class = shift;
  $extra_class = $extra_class ? " $extra_class" : '';
  $attrs = " $attrs" if $attrs;
  my $filename_ = $this->{'base_filename'};
  $filename_ =~ s/-\d+//g;
  $filename_ =~ s/-/_/g;
  return qq(<table class="$filename_$extra_class" align=center cellspacing=0$attrs>\n);
  }

=item $html = $log->RenderRowsHTML();

Render stored HTML table cells and clear them.

=cut

sub RenderRowsHTML ($) {
  my $this = shift;
  my $html = '';
  $html .= $this->RenderSectionHTML('title_html', 'th');
  $html .= $this->RenderSectionHTML('data_html', 'td');
  $this->{'title_html'} = [];
  $this->{'data_html'} = [];
  return $html;
  }

=item $text = $log->RenderRowsText();

Render stored text table cells and clear them.

=cut

sub RenderRowsText ($) {
  my $this = shift;
  my $text = '';
  my $lengthsp = $this->MeasureColumns();
  $text .= $this->RenderSectionText('title_text', $lengthsp);
  $text .= $this->RenderSectionText('data_text', $lengthsp);
  $this->{'title_text'} = [];
  $this->{'data_text'} = [];
  return $text;
  }

=item $html = $log->RenderSectionHTML($key, $tag);

Used internally to render either a group of header or data rows into HTML.

=cut

sub RenderSectionHTML ($$$) {
  my $this = shift;
  my $key = shift;
  my $tag = shift;
  my $html = '';
  my $rows = $this->{$key};
  return '' unless @$rows;

  for (my $rowi = 0; $rowi <= $#$rows; $rowi += 2) {
    my $metadata = $rows->[$rowi];
    my $attributes = $metadata->{'attributes'};
    my $classes = $metadata->{'classes'};
    my $row_class = $metadata->{'row_class'};
    if ($tag eq 'th' && !defined $row_class) {
      $row_class = 'top' . (($rowi/2)+1);
      }
    my $row = $rows->[$rowi+1];
    $html .= (defined $row_class) ? qq(<tr class="$row_class">) : '<tr>';
    for my $i (0..$#$row) {
      my $attribute = $attributes->[$i];
      $attribute = '' unless defined $attribute;
      my $class = $classes->[$i];
      my $cell = $row->[$i];
      $cell = '&nbsp;' unless (defined $cell) && length($cell); # required for Windows browsers
      $html .= qq(<$tag);
      $html .= qq( class="$class") if defined $class;
      $html .= qq( $attribute) if length($attribute);
      $html .= qq(>$cell</$tag>);
      }
    $html .= "</tr>\n";
    }

  return $html;
  }

=item $html = $log->RenderSectionText($key, $lengths);

Used internally to render either a group of header or data rows into text.

=cut

sub RenderSectionText ($$$) {
  my $this = shift;
  my $key = shift;
  my $lengthsp = shift;
  my $text = '';
  my $rows = $this->{$key};
  return '' unless @$rows;

  for (my $rowi = 0; $rowi <= $#$rows; $rowi += 2) {
    my $classes = $rows->[$rowi];
    my $row = $rows->[$rowi+1];
    for my $i (0..$#$row) {
#     if (ref($classes) eq 'HASH') { die "$key:".join(',', %$classes); }
      my $value = $row->[$i];
      $value = '' unless defined $value;
      $text .= sprintf(MakeFormat($lengthsp->[$i], $classes->[$i]), $value);
      $text .= ' ' unless $i == $#$row;
      }
    $text =~ s/\s+$//;
    $text .= "\n";
    }

  return $text;
  }

=item $html = $log->RenderText();

Render entire log (header, table and footer) as text and reset table data.

=cut

sub RenderText ($) {
  my $this = shift;
  my $html = '';
  $html .= $this->RenderHeaderText($this->{'title'},
    $this->{'base_filename'}, $this->{'options'});
  $html .= $this->RenderRowsText();
  $html .= $this->RenderFooterText();
  return $html;
  }

=item $log->Write($text,$html)

Write text to logs and console.

=cut

sub Write ($$$) {
  my $this = shift;
  my $text = shift;
  my $html = shift;
  my $fh;
  ExpandEntities $text;
  PrintConsoleCrossPlatform $text unless $this->{'options'}{'noconsole'};
# unless (defined $html) { use Carp; confess "no html"; }
  $fh = $this->{'text_fh'}; print $fh $text if $fh;
# use PerlIO; die join(',',PerlIO::get_layers($this->{'subhtml_fh'}));
  $fh = $this->{'subhtml_fh'}; print $fh $html if $fh;
  }

=item $log->WritePartialWarning($colspan);

Warn user that results are incomplete.

=cut

sub WritePartialWarning ($$) {
  my $this = shift;
  my $colspan = shift;
  $this->Write("Based on PARTIAL results\n\n", 
    "<tr><td colspan=$colspan class=warning>Based on PARTIAL results</td></tr>\n")
  }

=item $log->WriteRow(\@coltext[, \@colhtml]);

Appends a row of data cells, to be output later by Flush(),
possibly invoked by Close().  
Saves a copy of the current values of column classes and attributes.

=cut

sub WriteRow ($$;$) {
  my $this = shift;
  my $textp = shift;
  my $htmlp = shift;
  if (!defined $htmlp) { $htmlp = $textp; }
  if (@$textp) {
    for my $text (@$textp) { ExpandEntities $text; }
    push(@{$this->{'data_text'}}, [@{$this->{'column_classes'}}], [@$textp]) 
    }
  push(@{$this->{'data_html'}}, 
    {
      'attributes' => [@{$this->{'column_attributes'}}],
      'classes' => [@{$this->{'column_classes'}}],
      'row_class' => $this->{'row_class'},
    }, [@$htmlp]) 
    if @$htmlp;
  }

=item $log->WriteTitle(\@text, \@html);

Appends a rows of title cells to the current title area.
If data rows have already been stored, flush them and purge
any previous title rows.
Saves a copy of the current values of column classes and attributes.

=cut

sub WriteTitle ($$;$) {
  my $this = shift;
  my $textp = shift;
  my $htmlp = shift;
  if ($this->Flush()) {
    $this->{'title_html'} = [];
    $this->{'title_text'} = [];
    }
  if (!defined $htmlp) { $htmlp = $textp; }
  if (defined @$textp) {
    for my $text (@$textp) { ExpandEntities $text; }
    push(@{$this->{'title_text'}}, [@{$this->{'column_classes'}}], [@$textp]) 
    }
  push(@{$this->{'title_html'}}, 
    {
      'attributes' => [@{$this->{'column_attributes'}}],
      'classes' => [@{$this->{'column_classes'}}],
      'row_class' => $this->{'row_class'},
    }, [@$htmlp]) 
    if @$htmlp;
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

