#!/usr/bin/perl

# Copyright (C) 2006 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Server;

use strict;
use warnings;

use File::Spec;
use HTTP::Message qw(EncodeEntities);
use HTTP::Server;
use TSH::Config;
use TSH::Command::RoundRATings;
use TSH::Command::ShowPairings;
use TSH::Server::Content;
use Net::Domain qw(hostfqdn);

our (@ISA) = qw(HTTP::Server Exporter);
our (@EXPORT_OK) = qw(HTMLFooter HTMLHeader);

=pod

=head1 NAME

TSH::Server - provide web access to the tsh process

=head1 SYNOPSIS

  my $server = new TSH::Server($port) or die;
  $server->Start() or die;
  while ($server->Run()) {}
  $server->Stop();
  
=head1 ABSTRACT

This subclass of HTTP::Server provides single-threaded access to
the tsh process.

=head1 DESCRIPTION

=over 4

=cut

sub new ($@);
sub ConfigItem ($);
sub Render ($$);
sub ContentConfig ($$;$);
sub ContentDivision ($$$$);
sub ContentDocumentation ($$$$$);
sub ContentReport ($$$$$$);
sub ContentRoot ($$);
sub ContentSelfServiceScoring ($$);
sub ContentStaticReport ($$$$$);
sub HTMLFooter ();
sub HTMLHeader ($$$;$);
sub RenderSSSForm ($$);
sub SaveConfigChanges ($$);
sub Start ($);
sub Tournament ($);
sub ValidatePlayerEntry ($$);

=item $server = new TSH::Server($port);

Create a new TSH::Server object.  

C<port>: port to listen on for new connections.

=cut

sub new ($@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $port = shift;
  my $tournament = shift;
  my $config = $tournament->Config();
  my $logfh;
  if ($tournament->Virtual()) {
    $logfh = *STDERR;
    }
  else {
    my $logfn = $config->MakeRootPath("http$port.txt");
    open $logfh, ">$logfn" or die "Can't create $logfn: $!";
    # TODO: what encoding is this in?
    {
      my $save = select $logfh;
      $| = 1;
      select $save;
    }
    }
  my $this = $class->SUPER::new(
    'log_handle' => $logfh,
    'port' => $port,
#   'content' => \&TSH::Server::Content,
    'version' => "tsh/$main::gkVersion",
    );
  $this->{'tournament'} = $tournament;
  bless($this, $class);
  $this->SetContentHandler(sub { $this->Render(@_); }); # a closure
  $this->{'commands'}{'rrat'} = new TSH::Command::RoundRATings('noconsole' => 1);
  $this->{'commands'}{'sp'} = new TSH::Command::ShowPairings('noconsole' => 1);
  return $this;
  }

=item $error = TSH::Server::ConfigCheck();

Checks to see if the currently loaded configuration can be used to
run tsh in server mode, returns an error message if not.

=cut

sub ConfigCheck () {
  my @problems;
  unless ($config::max_rounds) {
    push(@problems, "The mandatory max_rounds option has not been specified.");
    }
  if (@problems) {
    if (@problems == 1) {
      return "You must correct this configuration before you can proceed. "
        . $problems[0];
      }
    else { die; }
    }
  else { return ''; }
  }

=item $html = TSH::Server::ConfigItem($name);

Returns a line of $html code for use in ContentConfig.

=cut

sub ConfigItem ($) {
  my $name = shift;
  my $value = eval "\$config::$name";
  my $type = TSH::Config::UserOptionType($name);
  my $help = TSH::Config::UserOptionHelp($name);
  if ($type eq 'internal') { return ''; }
  my $html = "<tr><td class=name>$name</td>";
  if (!defined $type) {
    $html .= "<td>unknown config item</td>";
    }
  if ($type eq 'boolean') {
    my $checked = $value ? ' checked': '';
    $html .= qq(<td class=value><input name=$name type=checkbox value=1$checked></td>);
    }
  elsif ($type eq 'integer') {
    $value = 0 unless $value;
    $html .= qq(<td class=value><input name=$name type=text maxlength=10 size=10 value="$value"></td>);
    }
  elsif ($type eq 'string') {
    $value = '' unless defined $value;
    $value = EncodeEntities $value;
    $html .= qq(<td class=value><input name=$name type=text maxlength=256 size=40 value="$value"></td>);
    }
  else {
    $html .= "<td>unknown config type: $type</td>";
    }
  if ($help) {
    $html .= "<td class=help>$help</td>";
    }
  $html .= "</tr>";
  return $html;
  }

=item $response = $server->Render($request);

Generate the content needed to serve an HTTP request.
See HTTP::Server for details.

=cut

sub Render ($$) {
  my $this = shift;
  my $request = shift;
  my $client = shift;
  my $url = $request->URL();

  my $renderer = new TSH::Server::Content($this, $request, $url, $client);
  return $renderer->Render();
  }

=item $response = $server->ContentConfig($request[, error]);

Generate the content needed to serve a request for '/config/index.html'.

=cut

sub ContentConfig ($$;$) {
  my $this = shift;
  my $request = shift;
  my $error = shift; $error = ref($error) ? $error->{'error'} : '';
  my $formp = $request->FormData();
  my $html = $this->HTMLHeader('Configure', 'event') . "<h1>tsh Configuration</h1>";
  if ($formp->{'post'}) {
    $html .= $this->SaveConfigChanges($formp);
    $error = ConfigCheck();
    }
  if ($error) {
    $html .= "<div class=failure>$error</div>";
    }
  $html .= <<"EOF";
<p class=p1>The following options control <cite>tsh</cite>&rsquo;s behaviour.
If you want to change them, click or type appropriately within the table,
then click on the Save Changes button below it.
</p>
<form method="post" action="/config/" enctype="application/x-www-form-urlencoded">
<table class=config align=center>
<tr><th>Option Name</th><th>Value</th><th>Description</th></tr>
EOF
  for my $key (sort(TSH::Config::UserOptions())) { $html .= ConfigItem $key; }
  $html .= <<"EOF";
</table>
<p class=p2><input type=submit name=post value=\"Save Changes\"></p>
</form>
<p class=p1>The following table lists the player divisions in your tournament and the name of the file in which each division&rsquo;s information is stored.</p>
EOF
  # autopair
  $html .= HTMLFooter();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => $html,
    );
  }

=item $response = $server->ContentDivision($request, $url, $division);

Generate the content needed to serve a request for '/division/DIVNAME'.

=cut

sub ContentDivision ($$$$) {
  # TODO: do some sort of caching here?
  my $this = shift;
  my $request = shift;
  my $url = shift;
  my $div = TSH::Division::CanonicaliseName(shift);
  my $html = $this->HTMLHeader('Main', 'event');
  my $tournament = $this->{'tournament'};
  my $config = $tournament->Config();
  my (@dps) = $tournament->Divisions();
  my $dp = $tournament->GetDivisionByName($div) || $dps[0];
  my $max_rounds = $dp->MaxRound0() + 1;
  $div = $dp->Name();
  $html .= "<div class=divlist>";
  $html .= "<span class=label>Division:</span>";
  for my $adp ($tournament->Divisions()) {
    my $adiv = $adp->Name();
    if ($div eq $adiv) {
      $html .= qq(<span class=here>$adiv</span>);
      }
    else {
      $html .= qq(<span class=there><a href="/division/$adiv/">$adiv</a></span>); 
      }
    }
  $html .= "</div>";
  $html .= qq(<form method="post" action="/division/$div" enctype="application/x-www-form-urlencoded">);
  $html .= qq(<table class=dedit cellspacing=0>);
  {
    my $start = "<td colspan=2 rowspan=3 class=reports>Reports</td>";
    my (@data) = (
      'Alpha Pair', 'alpha-pairings',
      'Rank Pair', 'pairings',
      'Standings', 'ratings',
      );
    while (@data) {
      my $label = shift @data;
      my $type = shift @data;
      $html .= "<tr>$start";
      $start = '';
      for my $round (1..$max_rounds) {
	$html .= sprintf(qq(<th class=report><a target="report" href="/%s-%s-%d.html">%s</a></th>), $div, $type, $round, $label); 
	}
      $html .= "</tr>";
      }
  }
  $html .= "<tr>"
    . qq(<th class=id>ID</th>)
    . qq(<th class=name>Name</th>);
  for my $round (1..$max_rounds) {
    $html .= qq(<th class=round>Round $round</th>); 
    }
  $html .= "</tr>";
  my (@players) = $dp->Players();
  my $pfmt = '%0' . length(scalar(@players)) . 'd';
  for my $pp (@players) {
    my $id = $pp->ID();
    my $name = $pp->Name();
    $html .= sprintf(qq(<tr><td class=id>$pfmt</td><td class=name>%s</td>),
      $id, $name);
    for my $round (1..$max_rounds) {
      my $round0 = $round - 1;
      my $ms = $pp->Score($round0);
      my $p12 = '';
      if ($config->Value('track_firsts')) {
	$p12 = ('','1st vs. ','2nd vs. ','draw vs. ','')[$pp->First($round0)||0];
        }
      my $oid = $pp->OpponentID($round0);
      my $on = '';
      if ($oid) { 
	$oid = sprintf($pfmt, $oid);
	my $op = $pp->Opponent($round0);
	if ($op) {
	  $on = $pp->Opponent($round0)->Initials();
	  }
        }
      else { $oid = '-'; }
      my $os = $pp->OpponentScore($round0);
      my $scores;
      if ((defined $ms) && (defined $os)) {
	$scores = "$ms&ndash;$os";
        }
      elsif ($oid ne '-') {
	$scores = ((defined $ms) ? $ms : '?') . '-' . 
	  ((defined $os) ? $os : '?');
        }
      elsif (defined $ms) {
	$scores = sprintf("%+d", $ms);
	$oid = 'bye';
        }
      else {
	$on = '??';
	$scores = '?';
        }
      $html .= qq(<td class=round><div class=opp>$p12$oid$on</div><div class=score>$scores</div></td>);
      }
    $html .= qq(</tr>);
    }
  $html .= qq(</table>);
  $html .= qq(</form>);
  $html .= HTMLFooter();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => $html,
    );
  }

=item $response = $server->ContentDocumentation($request,$url,$fname,$fext);

Generate the content needed to serve a request for a documentation file.

=cut

sub ContentDocumentation ($$$$$) {
  my $this = shift;
  my $request = shift;
  my $url = shift;
  my $fname = shift;
  my $fext = shift;

  return HTTP::Server::StaticContent("doc/$fname", $fext);
  }

=item $response = $server->ContentJavaScript($request,$url,$fname,$fext);

Generate the content needed to serve a request for a documentation file.

=cut

sub ContentJavaScript ($$$$$) {
  my $this = shift;
  my $request = shift;
  my $url = shift;
  my $fname = shift;
  my $fext = shift;

  return HTTP::Server::StaticContent("lib/js/$fname", $fext);
  }

=item $response = $server->ContentLibraryFile($request,$url,$fname,$fext);

Generate the content needed to serve a request for a documentation file.

=cut

sub ContentLibraryFile ($$$$$) {
  my $this = shift;
  my $request = shift;
  my $url = shift;
  my $fname = shift;
  my $fext = shift;

  return HTTP::Server::StaticContent("lib/$fname", $fext);
  }

=item $response = $server->ContentPhoto($request,$url,$fname,$fext);

Generate the content needed to serve a request for a documentation file.

=cut

sub ContentPhoto ($$$$$) {
  my $this = shift;
  my $request = shift;
  my $url = shift;
  my $fname = shift;
  my $fext = shift;

  return HTTP::Server::StaticContent("lib/pix/$fname", $fext);
  }

=item $response = $server->ContentPrimaryRoot($request);

Generate the content needed to serve a request for '/'.

=cut

sub ContentPrimaryRoot ($$) {
  my $this = shift;
  my $request = shift;
  my $tournament = $this->{'tournament'};
  my $t = $tournament->Config()->Terminology({ map { $_ => [] } 
    qw(MainMenuTitle MainMenuWelcome MainMenuFirst MainMenuEvent MainMenuHelp MainMenuTerms)});
  my $html = $this->HTMLHeader('Main', 'main');
  $html .= <<"EOF";
<h1>$t->{'MainMenuTitle'}</h1>
<p class=p1>$t->{'MainMenuWelcome'}</p>
<ul>
<li>$t->{'MainMenuFirst'}</li>
<li>$t->{'MainMenuEvent'}</li>
<li>$t->{'MainMenuHelp'}</li>
</ul>
<p class=p1>$t->{'MainMenuTerms'}</p>
EOF
  $html .= HTMLFooter();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => $html,
    );
  }

=item $response = $server->ContentReport($request, $url,$division, $type, $round);

Generate the content needed to serve a request for 
'/DIVNAME-pairings-ROUND.html'
or
'/DIVNAME-alpha-pairings-ROUND.html'
.

=cut

sub ContentReport ($$$$$$) {
  my $this = shift;
  my $request = shift;
  my $url = shift;
  my $dname = TSH::Division::CanonicaliseName(shift);
  my $type = shift;
  my $round = shift;
  my $html = $this->HTMLHeader('Main', 'event');
  my $tournament = $this->{'tournament'};
  my $dp = $tournament->GetDivisionByName($dname);
  if ($type =~ /^(?:alpha-)?pairings$/) {
    $this->{'commands'}{'sp'}->Run($tournament, $round, $dp);
    }
  elsif ($type eq 'ratings') {
    $this->{'commands'}{'rrat'}->Run($tournament, $round, $round, $dp);
    }
  my $basefn = $tournament->Config()->MakeHTMLPath(sprintf("%s-%s-%03d",
    $dname, $type, $round));
  if (-r "$basefn.html") {
    return HTTP::Server::StaticContent($basefn, 'html');
    }
  else {
    return new HTTP::Message( 'status-code' => 404, 'body' => "The report you asked for is unavailable." );
    }
  }

=item $response = $server->ContentRoot($request);

Generate the content needed to serve a request for '/'.

=cut

sub ContentRoot ($$) {
  my $this = shift;
  my $request = shift;
  my $tournament = $this->{'tournament'};
  if ($tournament->Config()->Value('primary_gui')) {
    return $this->ContentPrimaryRoot($request);
    }
  elsif ($tournament->CountDivisions() == 0) 
    { return $this->ContentConfig($request); }
  else { 
    return $this->ContentDivision($request, '/ignored',
      ($tournament->Divisions())[0]->Name()); 
    }
  }

=item $response = $server->ContentSelfServiceScoring($request);

Generate the content needed to serve a request for '/cgi/sss'.

=cut

sub ContentSelfServiceScoring ($$) {
  my $this = shift;
  my $request = shift;
  my $formp = $request->FormData();
  my $html = $this->HTMLHeader('Player Score Entry', 'event') . "<h1>Self-Service Scoring</h1>";
  if ($formp->{'validate'}) {
    $html .= $this->ValidatePlayerEntry($request);
    }
  else {
    $html .= $this->RenderSSSForm({'request' => $request});
    }
  $html .= <<"EOF";
</table>
</form>
EOF
  $html .= HTMLFooter();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => $html,
    );
  }

sub ContentShutdown ($) {
  my $this = shift;
  $this->Shutdown();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => '<html><head></head><body><p>Close this window to return to the <cite>tsh</cite> command line.</p></body></html>',
    )
  }

=item $response = $server->ContentStaticReport($request,$url,$fname,$fext);

Generate the content needed to serve a request for a 
report that tsh has already generated.

=cut

sub ContentStaticReport ($$$$$) {
  my $this = shift;
  my $request = shift;
  my $url = shift;
  my $fname = shift;
  my $fext = shift;

  return HTTP::Server::StaticContent($this->{'tournament'}->Config()->MakeHTMLPath($fname), $fext);
  }

=item $html = TSH::Server::HTMLFooter();

Return an HTML footer.

=cut

sub HTMLFooter () {
  return "</body></html>";
  }

=item $html = $s->HTMLHeader($section, $type);

Return an HTML header suitable for displaying in section $section.

=cut

my (@navbardata);
BEGIN {
  (@navbardata) = (
    {'label' => 'Main', 'url' => '/', 'types' => ['main','events', 'event'], 'active' => 0},
    {'label' => 'Events', 'url' => '/events/', 'types' => ['main','events', 'event'], 'active' => 1},
    {'label' => 'Configure', 'url' => '/config/', 'types' => ['events', 'event'], 'active' => 0},
    {'label' => 'Reports', 'url' => '/report/', 'types' => ['events', 'event'], 'active' => 0},
    {'label' => 'SSS', 'url' => '/cgi/sss', 'types' => ['events', 'event'], 'active' => 0},
    {'label' => 'Valet', 'url' => '#', 'types' => ['event'], 'onclick' => 'ViewValet(); return false', 'active' => 1},
    {'label' => 'Help', 'url' => '/doc/', 'types' => ['main', 'events', 'event'], 'active' => 1},
    {'label' => 'Tutorial', 'url' => '/doc/tutorial.html', 'types' => ['main', 'events', 'event'], 'active' => 0},
    {'label' => 'Quit', 'url' => '/shutdown/index.html', 'types' => ['main', 'events', 'event'], 'active' => 1},
    );
  for my $d (@navbardata) {
    $d->{'types'} = { map { $_ => 1 } @{$d->{'types'}} };
    }
  }

sub HTMLHeader ($$$;$) {
  my $this = shift;
  my $section = shift;
  my $type = shift;
  my $tourney = shift;
  my $config = $tourney && $tourney->Config();
  my $html = <<"EOF";
<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01//EN">
<html>
<head>
<meta name="viewport" content="width=device-width; user-scalable=false" />
<title>tsh $main::gkVersion</title>
<link rel=stylesheet href="/tsh.css">
</head>
<body class=gui>
EOF
  $html .= qq(<div class="navbar noprint">);
  $html .= "<span class=label>tsh $::gkVersion</span>";
  $html .= "<span class=colon>:</span> <span class=event>"
    . ($config->Value('event_name') || 'Unnamed Event')
    . ' (' . ($config->Value('event_date') || 'Unknown Date') . ')'
    . "</span> "
    if $config && $type ne 'main';
# $html .= qq(<a href="notyet">change event</a>);
  $html .= "<div class=seclist>";
  for my $nbd (@navbardata) {
    next unless $nbd->{'types'}{$type};
    my $label = $nbd->{'label'};
    $label = 'Return to <cite>tsh</cite> command line' if $label eq 'Quit' && $this->{'tournament'}->Virtual();
    if ($label eq $section) 
      { $html .= "<span class=here>$label</span>"; }
    elsif ($nbd->{'active'})
      { my $onclick = $nbd->{'onclick'} ? qq( onclick="$nbd->{'onclick'}") : ''; $html .= qq(<span class=there id="nav_\L$nbd->{'label'}\E"><a href="$nbd->{'url'}"$onclick>$label</a></span>); }
    else 
      { $html .= "<span class=inactive>$label</span>"; }
    }
  $html .= "</div>";
  $html .= "</div>";
  return $html;

  die "This stuff needs a home";
#   $html .= "</tr>";
#   $html .= <<"EOF";
# <script language="JavaScript">
# function Division (div) {
#   alert(div);
#   }
# </script>
# EOF
  }

=item $html = $server->RenderSSSForm(\%options);

Render an SSS form given the following options in C<\%options>.

C<errors>: two-element list of errors to display below table columns

=cut

sub RenderSSSForm ($$) {
  my $this = shift;
  my $options = shift;
  my $request = $options->{'request'};
  my $formp = $request->FormData();
  my $html = '';

  my $errors = '';
  if (my $ep = $options->{'errors'}) {
    $errors = "<tr class=error><th class=label>&nbsp;</th><td>$ep->[0]</td><td>$ep->[1]</td></tr>\n";
    $errors .= "<tr><td colspan=3>Please correct the errors above and resubmit your scores.</td></tr>\n";
    }
  my $config = $this->{'tournament'}->Config();
  my (@labels) = 
    ($config->Value('track_firsts') && !$config->Value('assign_firsts'))
      ? ('Played First','Played Second')
      : ('Winner', 'Loser');
  my %param;
  for my $key (qw(p1s p2s p1p p2p)) {
    my $value = $formp->{$key};
    $param{$key} = (defined $value) ? $value : '';
    $value =~ s/\W//g;
    }
  $html .= <<"EOF";
<form method="post" action="/cgi/sss" enctype="application/x-www-form-urlencoded">
<table class=pe align=center width=400px cellspacing=0>
<tr><th>&nbsp;</th><th>$labels[0]</th><th>$labels[1]</th></tr>
<tr>
<th class=label>Score</th>
<td class=score><input name=p1s type=text maxlength=5 size=8 value="$param{'p1s'}"></td>
<td class=score><input name=p2s type=text maxlength=5 size=8 value="$param{'p2s'}"></td>
</tr>
<tr>
<th class=label>Password</th>
<td class=pswd><input name=p1p type=password maxlength=8 size=8 value="$param{'p1p'}"></td>
<td class=pswd><input name=p2p type=password maxlength=8 size=8 value="$param{'p2p'}"></td>
</tr>
$errors<tr>
<td colspan=3>
<p class=p2>
If this is your first time using self-service scoring, please follow
the instructions carefully.
</p>
<ol>
<li>
Either player enters the game scores.  Click or use the tab or shift-tab
keys to move between boxes.
</li>
<li>
To confirm that each score is correct, each player enters their
password in the box below their own score.
</li>
<li>
Click on &ldquo;Submit Scores&rdquo;. You will be given a chance to confirm that your cumulative
results look correct before the scores are posted.
</li>
</ol>
</td>
</tr>
<tr>
<td colspan=3 align=center><input type=submit name=validate value=\"Submit Scores\"></p>
</tr>
</table>
EOF
  }

=item $html = $server->SaveConfigChanges($formp);

Saves changes to tsh's configuration as indicated by the form variables
in the hash %$formp.  Returns a message informing the user whether or
not the operation succeeded.

=cut

sub SaveConfigChanges ($$) {
  my $this = shift;
  my $formp = shift;
  my $fh;
  my $changed = 0;
  for my $key (TSH::Config::UserOptions()) {
    my $newvalue = $formp->{$key};
    my $type = TSH::Config::UserOptionType($key);
    my $error = UserOptionValidate($key, $newvalue);
    return "<div class=failure>$error</div>" if $error;
    my $oldvaluep = $config::{$key};
    if ($type eq 'boolean') { 
      if ($newvalue xor $$oldvaluep) {
	$changed = 1;
	$$oldvaluep = $newvalue ? 1 : 0;
        }
      }
    elsif ($type eq 'integer') { 
      $newvalue = $newvalue ? 0+$newvalue : 0; 
      if ($newvalue != ($$oldvaluep||0)) {
	$changed = 1;
	$$oldvaluep = $newvalue;
        }
      }
    elsif ($type eq 'string') { 
      $newvalue = (defined $newvalue) ? $newvalue : '';
      if ($newvalue ne ((defined $$oldvaluep) ? $$oldvaluep : '')) {
	$changed = 1;
	$$oldvaluep = $newvalue;
        }
      }
    else 
      { return "<div class=failure>Unknown key type ($type) for $key.</div>"; }
    }
  return $this->Config()->Save();
  }

=item $server->Start() or die;

Start the server by opening the listener socket.

=cut

sub Start ($) {
  my $this = shift;
  print "Starting server...\n";
  my $result = $this->SUPER::Start();
# my $host = hostfqdn(); # takes too long
  my $port = $this->{'port'};
  print <<"EOF" unless $this->{'tournament'}->Virtual();
tsh is now in server mode.  You can connect to the server at 
http://localhost:$port from this machine, or from other machines
by replacing 'localhost' with your machine's network address.
To start tsh in interactive mode, do not include a "config port" line
in your configuration file.
EOF
  }

sub Tournament ($) {
  my $this = shift;
  return $this->{'tournament'};
  }

sub UserTournament ($$;$) {
  my $this = shift;
  my $path = shift;
  my $new = shift;
  $path =~ s/^\.\/+//;
  $path =~ s/\/+$//;
  my $old = $this->{'user_tournaments'}{$path};
  if (defined $new) {
    $this->{'user_tournaments'}{$path} = $new;
    }
  return $old;
  }

=item $response = $server->ValidatePlayerEntry($request);

Generate the content needed to validate an SSS entry.

=cut

sub ValidatePlayerEntry ($$) {
  my $this = shift;
  my $request = shift;
  my $html = '<h1>Not yet</h1>';
  my $formp = $request->FormData();
  my $config = $this->{'tournament'}->Config();
  my (@passwords) = @$formp{qw(p1p p2p)};
  my (@scores) = @$formp{qw(p1s p2s)};
  my (@errors) = ('','');
  my $errors;
  my (@players);
  for my $i (0..1) {
    if (my $p = $config->GetPlayerByPassword($passwords[$i])) {
      $players[$i] = $p;
      }
    else {
      $errors[$i] = 'Invalid password';
      $errors++;
      }
   }
  if ($errors) {
    return $this->RenderSSSForm({'errors'=>\@errors, 'request' => $request});
    }

  $html .= "@passwords @scores";

  return $html;
  }

=back

=cut

1;

